/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

		Perform the trust propagation pass over a keyring
		
        $Id: pgpTrustProp.c,v 1.36.2.1 2001/04/23 22:47:28 hal Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"

#include <time.h>
#include <stddef.h>

#include "pgpDebug.h"
#include "pgpMakeSig.h"
#include "pgpMem.h"
#include "pgpEnv.h"
#include "pgpPktByte.h"
#include "pgpRegExp.h"
#include "pgpKeyPriv.h"
#include "pgpTimeDate.h"
#include "pgpTrustPriv.h"
#include "pgpHashPriv.h"
#include "pgpErrors.h"
#include "pgpPubKey.h"
#include "pgpSig.h"
#include "pgpSigSpec.h"
#include "pgpUsuals.h"
#include "pgpTypes.h"
#include "pgpX509Priv.h"

#ifndef NULL
#define NULL 0
#endif

/*
 * This symbol marks code which prints the progress of the maintenance
 * pass.  That code assumes stdio, which is not portable across GUIs, so
 * something will need to be done about it.  The prints are left in the
 * source to make the location of the replacements obvious.  The code
 * assumes a non-existent "FILE *f" to print to, so just #defining this
 * symbol will NOT work.
 */
#undef PRINT_PROGRESS

/*
 * The trust models...
 * All models start from a number of axiomatic "the buck stops here"
 * keys.  These are the keys owned by the PGP user.  This is the
 * initial set of trusted keys.  Each key has a trust value - a
 * level of confidence that the user places in signatures made by
 * that key.
 * 
 * -- The new trust model
 *
 * This is based on a much more rigorous treatment of trust by Ueli
 * Maurer.  See "Modelling a Public-Key Infrastructure", Ueli Maurer,
 * Proceedings 1996 European Symposium on Research in Computer Security
 * (ESORICS '96), E. Bertino (Ed.), Lecture Notes in Computer Science,
 * Berlin: Springer-Verlag, Rome, Italy, Sept. 1996.  Also available from
 * <URL: http://www.inf.ethz.ch/personal/maurer/publications>.
 *
 * Consider the graph formed by taking each key as a node and
 * each signature as an edge, and assign each signature an independent
 * probability of being correct.  (The probabilities don't have to be
 * independent, but it gets messy fast if they aren't, so the possibility
 * isn't considered any further.)  For each possible combination of
 * valid and invalid signatures, assign it a probability based on the
 * probability of each signature being correct, and if there is a path
 * of valid signatures from the root set to a given userid, it is correct.
 * Add up the probabilities of all the cases in which the userid is correct,
 * and you have the computed probability that the userid is correct.
 *
 *
 * This implements the basic Maurer trust model fairly literally.  The
 * Maurer model is often considered to be "intractably complex" (an
 * out of context quote from his paper), but with certain
 * simplifications it should be feasible:
 *
 * - Focus on on-the-fly validation so we only have to find the validity of
 *   a single key.
 * - Search for paths from the target key back to an axiomatic key such that
 *   all keys on the path are trusted introducers (nonzero confidence).
 * - Limit this search by a depth constraint, and possibly limit the total
 *   number of nodes we will look at if the depth constraint is inadequate.
 * - Find the basic trust contribution of each of the resulting paths (not yet
 *   considering interactions) as the product of the trust of each segment on
 *   the path.
 * - Sort the paths by this value and prune to keep just the top 10 or
 *   so paths.
 * - Apply Maurer's algorithm, considering all subsets of paths (2**n of them,
 *   where n is the number of paths).  For each subset, form the union of all
 *   segments on the paths in that subset, and take the product of the segment
 *   trusts.  Subsets with an odd number of paths add confidence, those with an
 *   even number of paths subtract.
 *
 * By limiting ourselves to just the trusted introducers (plus one
 * other) the total number of keys to be considered should not become
 * too large.  Given practical and customary limitations on the number
 * of signatures a key will usually have, it should be possible to
 * find paths back to the axiomatic keys in a reasonable amount of
 * time, especially if the algorithm only needs to be run for a single
 * key (or a small number of keys) when we encrypt or check a
 * signature.  Then when we prune the list of paths to just the top
 * 10, we should still be able to get a reasonably accurate
 * (under)estimate of the actual trust, unless the key gained trust by
 * having a large number of paths each with only a small amount of
 * trust.  That case must be neglected for efficiency in this
 * algorithm.  Having pruned the paths to a small enough set we can
 * then apply Maurer's algorithm efficiently.
 *
 * There is an additional complication specific to the PGP trust
 * model.  Maurer's algorithm is expressed in terms of paths where
 * each segment has an associated trust level.  These trusts are
 * described in PGP as user-defined confidence levels specified for
 * the userid(s) on a key.  This means that the output of the
 * calculation above is a validity value for a particular userid on a
 * key, based on the signatures on that userid.  The inputs need to be
 * based on the user defined confidence values for the userids.  The
 * complication arises for keys which have more than one userid
 * with different user-defined confidence levels.  It is not clear how
 * to map this situation to Maurer's trust model.
 *
 * For keys with only one userid, the confidence of that userid is used.
 * If there is more than one userid, a conservative approach is to use
 * the userid with the smallest user-defined confidence value.  (Userids
 * for which the user has not specified a confidence level do not
 * enter into this calculation.)  If all of the userids (of those with
 * non-zero confidence) have validity values (from earlier runs of the
 * trust calculation) then we can choose the confidence of the userid
 * with the highest validity as in the previous trust model.
 *
 * Not all of the optimizations above are implemented in the current
 * code, nor has it been thoroughly tested.  In particular, the
 * trustprop function is still used and tries to run the algorithm on
 * all keys in the set.  Although a depth limitation is used there is
 * no other way of limiting the construction of the paths.
 *
 * Some further ideas for improvement:
 *
 * - Paths don't need to include the last step to the trusted
 *   introducer, since that always has confidence of 1.0.
 *
 * - For creating the path subset unions, a linear search for overlap
 *   is used now.  This may be OK for short paths but a more efficient
 *   algorithm would be better.  Perhaps a hash table for path segments
 *   in the subset would be worthwhile.
 *
 * - When adding/subtracting the pathconf values, maybe it would be
 *   better to accumulate 1-confidence rather than confidence?  I am
 *   worried about roundoff error for values very close to 1.0.
 *
 * - Could use a Gray code to go through path subsets in a sequence
 *   which would alternate even and odd parity subsets, so we would
 *   alternately add and subtract trust.  This might help with
 *   accuracy.  (A Gray code is a binary counter which has the property
 *   that only one bit changes each step.  One example is:
 *   ++j, i ^= j & (j ^ (j-1)); which counts j as a regular counter and
 *   i as a Gray code counter.)
 */


static PGPBoolean
sKeyHasConfidence (PGPKeyDBObj *key);


/********************** REGEXP Support ***************************/


/* Clear key regexp pointers */
	static void
mntClearKeyRegexps( PGPKeyDB *db )
{
	PGPKeyDBObj *key;
	PGPKeyInfo *kinfo;

	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		kinfo = pgpKeyToKeyInfo( key );
		kinfo->regexp = NULL;
	}
}




/* Check whether a sig is made illegal due to regexp considerations.
 * return TRUE if OK, FALSE if not.  If we have some error handling the regexp,
 * we say the sig is not OK.
 */
	static PGPBoolean
mntSigOKWithRegexpOnSigner( PGPKeyDBObj const *sig, PGPKeyDBObj *signer )
{
	PGPKeyDBObj *parent;
	PGPKeyInfo *kinfo;
	char const *udata;
	PGPSize len;
	PGPSize hlen;
	int rslt;

	pgpAssert( OBJISSIG( sig ) );

	/* See if signing key got its trust conditioned with a regexp */
	if( !pgpKeyDBObjIsReal( signer ) )
		return TRUE;
	pgpAssert( OBJISKEY( signer ) );
	kinfo = pgpKeyToKeyInfo( signer );
	if( IsNull( kinfo->regexp ) )
		return TRUE;

	/* Can only handle sigs on userids for regexps */
	parent = sig->up;
	if( !OBJISUSERID(parent) )
		return FALSE;

	/* See if userid matches regexp */
	udata = (const char *) pgpFetchObject( parent, &len );
	hlen = pgpPktBufferHeaderLen( (PGPByte *) udata );

	rslt = pgpRegExec( (regexp *)kinfo->regexp, udata+hlen );
	if( rslt == 1 )
		return TRUE;
	return FALSE;
}

/* Entry point which finds signer from sinfo->by */
	static PGPBoolean
mntSigOKWithRegexp( PGPKeyDBObj const *sig )
{
	PGPKeyDBObj *signer;
	PGPSigInfo *sinfo;

	pgpAssert( OBJISSIG( sig ) );

	/* See if signing key got its trust conditioned with a regexp */
	sinfo = pgpSigToSigInfo( sig );
	signer = sinfo->by;
	return mntSigOKWithRegexpOnSigner( sig, signer );
}



/******************** Signature Utility Functions **********************/


/*
 * Check to see if the signature is valid at this point in time.  
 *
 * To avoid overflow problems with 65535 days being larger than a 32-bit
 * timestamp, compute how old the signature is in days, and compare
 * that with the validity period.  If its age, in days rounded down,
 * is < the validity period, it is valid.  To deal with a validity
 * period of 0 being infinite, subtract one from the validity using
 * unsigned arithmetic and change the < to <=.  (n < 5 iff n <= 4, for
 * integer n).  0 will wrap to UINT_MAX, and every unsigned number
 * compared <= UINT_MAX, so the test will always be true.
 *
 */
static PGPBoolean
mntSigIsValidTime(PGPUInt32 timenow, PGPUInt32 tstamp, PGPUInt32 validity)
{
	/* For dates in future, give 24 hour slop */
	if( timenow < tstamp )
		return  timenow + 24*3600 > tstamp;
	/* Else check for expiration */
	return (PGPUInt32)(timenow-tstamp) <= (validity-1u) ;
}

/* True if signature is a valid, checked, unexpired key signature. */
static PGPBoolean
mntSigIsValidTimeKeySig(PGPKeyDBObj const *sig, PGPUInt32 timenow)
{
	PGPSigInfo *sinfo;

	if( !OBJISSIG( sig ) )
		return FALSE;
	if( !pgpKeyDBObjIsReal( sig ) )
		return FALSE;
	sinfo = pgpSigToSigInfo( sig );
	return
		   (sinfo->trust & PGP_SIGTRUSTF_CHECKED)
		&& !(sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL)
		&& (sinfo->type & 0xF0) == PGP_SIGTYPE_KEY_GENERIC
		&& mntSigOKWithRegexp( sig )
		&& mntSigIsValidTime(timenow, sinfo->tstamp, sinfo->sigvalidity);
}
		
/*
 * Check if the sig is valid, that is, not expired, not in the future,
 * cryptographically checked, not revoked by CRL, and OK with regexp.
 */
static PGPBoolean
mntSigIsValid( PGPKeyDBObj const *sig, PGPUInt32 timenow)
{
	PGPSigInfo *sinfo;

	pgpAssert(OBJISSIG(sig));
	sinfo = pgpSigToSigInfo( sig );
	return (sinfo->trust & PGP_SIGTRUSTF_CHECKED)
		&& !(sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL)
		&& mntSigOKWithRegexp( sig )
		&& mntSigIsValidTime( timenow, sinfo->tstamp, sinfo->sigvalidity );
}

/*
 * Return the newest sibling signature, based on the nextby fields.
 * Advances *sig to point beyond the sigs on this object
 * Sig must be cryptographically valie, but it may be expired or revoked.
 */
static PGPKeyDBObj *
mntNewestSiblingSig(PGPKeyDBObj const **sig)
{
	PGPKeyDBObj const *newest;
	PGPSigInfo *siginfo;
	PGPSigInfo *newsiginfo;

	pgpAssert(OBJISSIG(*sig));
	siginfo = pgpSigToSigInfo( *sig );
	newest = (*sig);
	newsiginfo = siginfo;
	while ((*sig = siginfo->nextby) != NULL) {
		pgpAssert(OBJISSIG(*sig));
		siginfo = pgpSigToSigInfo( *sig );
		if( !pgpKeyDBObjIsReal( *sig ) )
			continue;
		if( (*sig)->up != newest->up )
			break;
		if( siginfo->trust & PGP_SIGTRUSTF_CHECKED
				&& newsiginfo->tstamp < siginfo->tstamp )
		{
			newest = (*sig);
			newsiginfo = siginfo;
		}
	}
	return (PGPKeyDBObj *)newest;
}


/* True if a non-exportable sig for which we don't have the secret */
static PGPBoolean
mntForeignSignature(PGPKeyDBObj const *sig)
{
	PGPKeyDBObj *key;
	PGPSigInfo *sinfo;
	PGPKeyInfo *kinfo;

	pgpAssert( OBJISSIG( sig ) );
	sinfo = pgpSigToSigInfo( sig );

	/* False if exportable */
	if( SIGISEXPORTABLE(sinfo) )
		return FALSE;

	key = sinfo->by;
	pgpAssert( pgpKeyDBObjIsReal( key ) );
	pgpAssert( OBJISKEY( key ) );
	kinfo = pgpKeyToKeyInfo( key );
	
	/* False if by axiomatic key */
	if( kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP )
		return FALSE;
	/* Or if by a secret key */
	if( pgpKeyIsSec( key ) )
		return FALSE;
	/* Else true, a non-exportable sig by a non-secret key */
	return TRUE;
}	



/********************** Trust Model 2 ***************************/

#define LOOKINGAT(key) (((key)->flags & KEYF_TRUSTTMP) != 0)
#define SETLOOKINGAT(key) ((key)->flags |= KEYF_TRUSTTMP)
#define CLEARLOOKINGAT(key) ((key)->flags &= ~KEYF_TRUSTTMP)

/*
 * Maximum number of paths we will keep.
 * Must calculate trust for each subset.
 */
#define TOTALPATHMAX 10


/*
 * The Path and PathList routines use a convention copied from the
 * PGPPipeline package for tail pointers.  The tail pointer points
 * at the "next" pointer of the last packet, or if there are no packets
 * it points at the "head" pointer for the whole list.  This allows a
 * uniform interface for adding packets whether they are the first or
 * not.
 */

#undef DEBUGPATH

#ifdef DEBUGPATH

static char const hexchar[16] = {
	'0','1','2','3','4','5','6','7',
	'8','9','A','B','C','D','E','F'
};

static int
pgpDebugPrintKeyID(FILE *f, PGPKeyDBObj *key)
{
	PGPKeyInfo *kinfo;
	PGPByte *buf;
	int i;

	pgpAssert (OBJISKEY(key));
	kinfo = pgpKeyToKeyInfo( key );
	buf = kinfo->keyID;
	for (i = 4; i < 8; i++) {
		putc(hexchar[buf[i] >> 4 & 15], f);
		putc(hexchar[buf[i] & 15], f);
	}
	return 8;
}

static int
pgpDebugPrintKeyUserID(FILE *f, PGPKeyDBObj *key)
{
	char buserid[256];
	PGPByte const *udata;
	PGPKeyDBObj *uid;
	size_t len, hlen;

	uid = key->down;
	while (IsntNull(uid) && !OBJISUSERID(uid)) {
		uid = uid->next;
	}
	if( IsntNull( uid ) ) {
		udata = pgpFetchObject( uid, &len );
		hlen = pgpPktBufferHeaderLen( udata );
		strncpy (buserid, (char *) udata+hlen, pgpMin(len-hlen, 255));
		buserid[len-hlen] = '\0';
		fputs (buserid, f);
	} else {
		fputs ("<null userid>", f);
	}
	return 0;
}

static int
pgpDebugPrintKey(FILE *f, PGPKeyDBObj *key)
{

	pgpDebugPrintKeyID (f, key);
	fprintf (f, "(");
	pgpDebugPrintKeyUserID (f, key);
	fprintf (f, ")");
	return 0;
}

static void
pgpDebugPrintPath(FILE *f, Path *path)
{
	fprintf (f, " Path ");
	while (path) {
		fprintf (f, "from ");
		pgpDebugPrintKey (f, path->src);
		fprintf (f, " to ");
		pgpDebugPrintKey (f, path->dest);
		fprintf (f, ", conf: %g\n", path->confidence);
		path = path->next;
		if( IsntNull(path) )
			fprintf (f, "  ...segment ");
	}
}

static void
pgpDebugPrintPathList(FILE *f, PathList *pl)
{
	fprintf (f, "Begin PathList\n");
	while (pl) {
		fprintf (f, " (conf=%g) ", pl->confidence);
		pgpDebugPrintPath (f, pl->path);
		pl = pl->next;
	}
	fprintf (f, "End PathList\n");
}

#endif /* DEBUGPATH */

static double pathConfidence (Path *path);



/* Allocate a Path segment */
static Path *
pathAlloc (PGPKeyDB *db)
{
	Path       *path = db->paths;

	if( IsntNull(path) ) {
		db->paths = path->next;
	} else {
		path = (Path *) memPoolAlloc (&db->pathpool, sizeof(Path),
			alignof(Path));
	}
	return path;
}

/* Free a Path segment */
static void
pathFree (Path *path, PGPKeyDB *db)
{
	if( IsNull(path) )
		return;
	path->next = db->paths;
	db->paths = path;
}

/* Free an entire Path */
static void
pathFreeAll (Path *phead, PGPKeyDB *db)
{
	Path **ptail;

	if( IsNull(phead) )
		return;
	ptail = &phead->next;
	while (*ptail) {
		ptail = &(*ptail)->next;
	}
	*ptail = db->paths;
	db->paths = phead;
}

/* Allocate a PathList element */
static PathList *
pathListAlloc (PGPKeyDB *db)
{
	PathList   *plist = db->pathlists;

	if( IsntNull(plist) ) {
		db->pathlists = plist->next;
		plist->next = NULL;
	} else {
		plist = (PathList *) memPoolAlloc (&db->pathpool,
					sizeof(PathList), alignof(PathList));
	}
	return plist;
}

#if 0
/* Free a PathList element, also freeing any path it points to */
static void
pathListFree (PathList *plist, PGPKeyDB *db)
{
	if( IsntNull(plist->path) )
		pathFreeAll (plist->path, db);
	plist->path = 0;
	plist->confidence = 0.;
	plist->next = db->pathlists;
	db->pathlists = plist;
}
#endif

/* Free a whole PathList, also freeing the Paths it points to */
static void
pathListFreeAll (PathList *plhead, PGPKeyDB *db)
{
	PathList **pltail = &plhead->next;
	while (*pltail)
		pltail = &(*pltail)->next;
	*pltail = db->pathlists;
	db->pathlists = plhead;
	while (plhead != *pltail) {
		pathFreeAll (plhead->path, db);
		plhead->path = 0;
		plhead->confidence = 0.;
		plhead = plhead->next;
	}
}


/* Add a Path segment to a Path
 *
 * Returns	updated Path tail pointer
 *
 * tail		tail pointer for Path to be added to
 * seg		pointer to Path segment to add
 *
 */
static Path **
pathAddSeg (Path **tail, Path *seg)
{
	seg->next = *tail;
	*tail = seg;
	return &seg->next;
}

/*
 * Make a copy of a Path, leaving the original untouched.
 *
 * Returns	pointer to new cloned path
 *
 * p1		pointer to the path to be cloned
 * db		KeyDB used for allocations of new path segments
 *
 */
static Path *
pathClone (Path *p1, PGPKeyDB *db)
{
	Path       *phead = NULL,
	          **ptail = &phead;

	while (p1) {
		*ptail = pathAlloc(db);
		if( IsNull(*ptail) )
			return NULL;
		**ptail = *p1;
		p1 = p1->next;
		ptail = &(*ptail)->next;
	}
	return phead;
}

/* Add a copy of a Path to the PathList
 *
 * Returns	tail pointer of updated PathList
 *
 * tail		tail pointer of PathList to be added to
 * path		pointer to Path to add to the PathList
 * db		PGPKeyDB to use for memory allocations
 *
 */
static PathList **
pathListAddPathClone (PathList **tail, Path *path, PGPKeyDB *db)
{
	PathList   *pl;

	pl = pathListAlloc(db);
	if( IsNull(pl) )
		return NULL;
	if( IsntNull(path) )
	{
		pl->path = pathClone (path, db);
		pl->confidence = pathConfidence (path);
	} else {
		pl->path = NULL;
		pl->confidence = 1.0;
	}
	pl->next = *tail;
	*tail = pl;
	return &pl->next;
}


/* Calculate the confidence associated with a path. */
static double
pathConfidence (Path *path)
{
	double     conf = 1.0;

	while (path) {
		conf *= path->confidence;
		path = path->next;
	}
	return conf;
}

/*
 * Prune the PathList to keep only the highest-confidence pathmax Paths.
 * Return the new PathList, free the no longer used Paths.
 *
 * Returns	new PathList with no more than pathmax paths
 *
 * plist	PathList to be pruned
 * pathmax	maximum number of paths to allow
 * db		PGPKeyDB to use for memory allocations
 *
 */
static PathList *
pathListPrune (PathList *plist, unsigned pathmax, PGPKeyDB *db)
{
	PathList   *iplist;			/* Copy of plist */
	PathList  **pl1,			/* Iterator over plist */
	          **maxpl1;			/* Max confidence pl1 pointer */
	PathList   *nplist,			/* New pathlist head */
	          **nptail;			/* New pathlist tail */
	unsigned    npaths,			/* Number of paths in plist initially */
	            newnpaths;		/* Number of paths in final nplist */
	double      maxconf;		/* Best confidence value so far */

	/* Calculate confidence value for each path */
	iplist = plist;
	npaths = 0;
	while (plist) {
		npaths += 1;
		pgpAssert( plist->confidence == pathConfidence (plist->path) );
		plist = plist->next;
	}
	plist = iplist;

	/* Initialize the new path list */
	nplist = plist;
	if( npaths > pathmax ) {
		nplist = NULL;
		nptail = &nplist;
		newnpaths = pgpMin (pathmax, npaths);
		while (newnpaths--) {
			/* Add the best path from plist to nplist */
			pl1 = &plist;
			maxconf = 0.;
			maxpl1 = pl1;
			while (*pl1) {
				/* Find best path in plist in **maxpl1 */
				if( (*pl1)->confidence > maxconf ) {
					maxconf = (*pl1)->confidence;
					maxpl1 = pl1;
				}
				pl1 = &(*pl1)->next;
			}
			*nptail = *maxpl1;
			*maxpl1 = (*maxpl1)->next;
			nptail = &(*nptail)->next;
			*nptail = NULL;
		}

		if( IsntNull(plist) )
			pathListFreeAll (plist, db);
	}
	return nplist;
}


/*
 * Append a (copy of a) second path to the first, removing common segments.
 * Return pointer to "next" pointer of last segment (and pass that as ptail)
 *
 * Return	tail pointer for updated Path
 *
 * phead	pointer to Path which gets appended to
 * ptail	tail pointer for Path which gets appended to
 * p2		pointer to Path to append (a copy of)
 * db		PGPKeyDB to use for memory allocations
 *
 */
static Path **
pathUnion (Path *phead, Path **ptail, Path *p2, PGPKeyDB *db)
{
	Path       *p1;
	Path      **iptail = ptail;

	/* Add path p2 but skip those segments in p1 */
	for ( ; p2; p2 = p2->next) {
		p1 = phead;
		for (p1=phead; p1 && p1 != *iptail; p1 = p1->next) {
			/* Eliminate path segments which are in there already */
			if( p2->src == p1->src && p2->dest == p1->dest ) {
				/* Make sure our confidence rules are working */
				pgpAssert (p2->confidence == p1->confidence);
				break;
			}
		}
		if( IsntNull(p1) && p1 != *iptail )
			continue;
		*ptail = pathAlloc(db);
		if( IsNull(*ptail) )
			return NULL;
		**ptail = *p2;
		ptail = &(*ptail)->next;
		*ptail = NULL;
	}
	return ptail;
}

/*
 * Calculate the confidence for a list of paths.  We add the confidence for
 * each path, subtrace the confidence for the unions of all pairs of paths,
 * add the confidence for the unions for all the triples, and so on.
 * We actually do this by taking each subset of the paths, unioning them,
 * calculating the confidence for that path, and adding or subtracting it
 * based on the parity.
 */
static double
pathListConfidence (PathList *plist, PGPKeyDB *db)
{
    unsigned        pathcount,
                    mask,
	                tmask;
    PathList       *iplist = plist;
    double          conf = 0.;

#ifdef DEBUGPATH
	fprintf (stdout, "Maurer alg:\n");
#endif

	pathcount = 0;
	while (plist) {
		++pathcount;
		plist = plist->next;
	}
	plist = iplist;

	pgpAssert (pathcount < sizeof(unsigned) * 8);
	for (mask=1; mask < (1U<<pathcount); ++mask) {
		double pathconf;
		Path *phead=NULL, **ptail=&phead;
		int oddparity = 0;
		plist = iplist;
		tmask = mask;
		while (plist && tmask) {
			if( tmask & 1 ) {
				ptail = pathUnion (phead, ptail, plist->path, db);
				oddparity = !oddparity;
			}
			plist = plist->next;
			tmask >>= 1;
		}
		pathconf = pathConfidence (phead);
#ifdef DEBUGPATH
		fprintf (stdout, "%sing %g: ", oddparity?" add":" subb",pathconf);
		pgpDebugPrintPath (stdout, phead);
#endif
		if( oddparity )
			conf += pathconf;
		else
 			conf -= pathconf;
		pathFreeAll (phead, db);
		phead = NULL;
		ptail = &phead;
	}
#ifdef DEBUGPATH
	fprintf (stdout, "Net of %g\n", conf);
#endif
	return conf;
}

/*
 * Find the confidence level to use for the given key.
 * This is difficult in this trust model because we generally don't
 * have validity on the userids associated with the key, since there is
 * no well defined (nor arbitrarily imposed) ordering of the graph.
 * The result is that when we do our calculation we may end up with a
 * de facto validity on a key/userid that has no relation to the (cached)
 * validity stored on the key.
 *
 * So we have a set of heuristics to choose the confidence associated
 * with some userid, to wit:
 *
 * If all the userids have at least some validity, we choose the
 * confidence associated with the most valid userid.  Otherwise we
 * just choose the lowest confidence of all the userids.  The reason
 * for the second case is so that if we have a true userid and a false
 * userid, but the true userid doesn't have validity yet, while the
 * false userid has a little tiny bit of validity, we don't want to
 * choose the confidence of the false userid, which might be very
 * high.  It might be that the next step in the path will give high
 * validity to the true (but low confidence) userid.  We would end up
 * counting this key as both high confidence and high validity, which
 * may be wron
 *
 * This heuristic might seem to have the problem that if you get a new userid
 * for a trusted
 * key its trust will go away until either the new userid gets some validity
 * or you set the confidence level on the new userid.  But this does not
 * happen because such userids are not included in the confident set.
 *
 * A worse problem is that Snidely could get Nell's userid on his key, get
 * some low-confidence signer to sign Nell, and get an even lower conf-
 * idence signer to sign his own.  Then we would use Nell's confidence
 * on Snidely's key, and if there were also a good signer on Snidely's userid,
 * which we hadn't run into yet, we would again count Snidely's signatures
 * too highly.
 *
 * So this suggests a safer heuristic which is to take either the lowest
 * confidence, or else perhaps to downgrade the confidence by the
 * validities.  However doing this if all the validities are accurate will
 * end up squaring them.  Maybe we could just downgrade by the relative
 * validities, best to worst...
 *
 * (For now we have an #if below to always use the lowest confidence)
 *
 * An idea suggested by Mark is to base the confidence on the particular
 * userid which is the next step in the path.  However this complicates the
 * union operation.
 * Either a path no longer has a well defined confidence, preventing
 * Maurer's algorithm from working, or else we have
 * to define a path as being from userid to userid, and if a key has two
 * userids (different email addresses, say, home and work) we would end up
 * counting its contribution to trust twice.  (This is essentially because
 * the assumption of independence fails spectacularly in this case!)
 *
 * Keep in mind that despite all these difficulties, if there is only one
 * userid then there is no problem!  We just use the confidence on that userid.
 */
static double
pathKeyConfidence (PGPKeyDBObj *key)
{
	PGPKeyDBObj *	userid;
	PGPKeyInfo *	kinfo;
	PGPUserIDInfo *	uinfo;
	double          minconfidence = 1.0,
	                confidence,
	                errorchance,
					useridconfidence,
					keyconfidence,
	                maxvalidconfidence = 0.;
	PGPUInt16       maxvalidity = 0;
	PGPBoolean		allvalid = TRUE;	/* True if all userids have validity */
	PGPBoolean		validuserid = FALSE;/* True if at least one valid userid */

	kinfo = pgpKeyToKeyInfo( key );
	for (userid = key->down; IsntNull(userid); userid = userid->next) {
		if( !pgpKeyDBObjIsReal( userid ) )
			continue;
		if( !OBJISUSERID(userid) )
			continue;
		uinfo = pgpUserIDToUserIDInfo( userid );
		if( uinfo->confidence == PGP_NEWTRUST_UNDEFINED )
			continue;
		validuserid = TRUE;
		errorchance =  pgpTrustToDouble (
		                    pgpTrustToIntern(uinfo->confidence));
		confidence = 1. - 1. / errorchance;
		if( confidence < minconfidence )
			minconfidence = confidence;
		if( uinfo->valid == 0 ) {
			allvalid = 0;
		} else if( uinfo->valid > maxvalidity ||
						(uinfo->valid == maxvalidity &&
						 confidence < maxvalidconfidence) ) {
			maxvalidity = uinfo->valid;
			maxvalidconfidence = confidence;
		}
	}
	/* If called with no valid userid, minconfidence is 0 */
	if( !validuserid )
		minconfidence = 0.;
	/*
	 * If all userids have known validity, use the most valid userid's
	 * confidence.  Else choose the lowest established confidence of
	 * any userid.
	 */
#if 0
/* Too risky to use this heuristic, just take minimum confidence */
	if( allvalid )
		useridconfidence = maxvalidconfidence;
	else
#endif
		useridconfidence = minconfidence;

	keyconfidence = 0.;
	if( kinfo->signedTrust != 0 ) {
		errorchance =  pgpTrustToDouble (
		                    pgpTrustToIntern(kinfo->signedTrust));
		keyconfidence = 1. - 1. / errorchance;
	}

	return pgpMax (keyconfidence, useridconfidence);
}

/* Prototype */
static PathList **
sFindPathsKey (PGPKeyDBObj *start, PathList **ppath,
	Path **predh, Path **predt, unsigned depth,
    unsigned maxdepth, PGPUInt32 timenow, PGPKeyDB *db, PGPKeyDB *altdb);

/* Helper routines for sFindPathsUserID */

/* Return the newest cryptograhpically-good signature by the given key on
 * the given userid.  This does not check for expirations or regexps.
 */
static PGPKeyDBObj *
sNewestGoodSig (PGPKeyDBObj *userid, PGPKeyDBObj *key)
{
	PGPKeyDBObj		*bestsig = NULL,
					*sig;
	PGPSigInfo		*sinfo;
	PGPSigInfo		*bestsiginfo = NULL;

	pgpAssert (OBJISUSERID(userid));
	pgpAssert (OBJISKEY(key));

	for (sig = userid->down; IsntNull(sig); sig = sig->next) {
		if( !pgpKeyDBObjIsReal( sig ) )
			continue;
		if( OBJISSIG(sig) ) {
			sinfo = pgpSigToSigInfo( sig );
			if ((sinfo->trust & PGP_SIGTRUSTF_CHECKED) && sinfo->by == key ) {
				/* Good sig by key, see if it is newer than bestsig */
				if( IsNull(bestsig)
					|| bestsiginfo->tstamp < sinfo->tstamp ) {
					bestsig = sig;
					bestsiginfo = sinfo;
				}
			}
		}
	}
	return bestsig;
}

/*
 * Return true if there is a signature by the key on an earlier sibling
 * of this userid.  This alerts us to a condition which could cause duplicate
 * paths, which will waste resources.  key and localkey will be distinct
 * in the case where signer is in an alternate keydb.  In that case localkey
 * is the key in the sig's db and key is the key in the alternate db.
 * Otherwise they are the same.
 */
static int
sEarlierValidSig (PGPKeyDBObj *userid, PGPKeyDBObj *key,
	PGPKeyDBObj *localkey, PGPUInt32 timenow)
{
	PGPKeyDBObj		*parent,
					*sibuserid,
					*sig;
	PGPSigInfo		*sinfo;

	pgpAssert (OBJISUSERID(userid));
	pgpAssert (OBJISKEY(key));
	pgpAssert (OBJISKEY(localkey));

	parent = userid->up;
	pgpAssert (OBJISKEY(parent));

	for (sibuserid=parent->down; sibuserid!=userid;
		 							sibuserid=sibuserid->next) {
		if( !pgpKeyDBObjIsReal( sibuserid ) )
			continue;
		if( !OBJISUSERID(sibuserid) )
			continue;
		for (sig=sibuserid->down; IsntNull(sig); sig=sig->next) {
			if( !pgpKeyDBObjIsReal( sig ) )
				continue;
			if( OBJISSIG(sig) ) {
				sinfo = pgpSigToSigInfo( sig );
				if( (sinfo->trust & PGP_SIGTRUSTF_CHECKED)
						&& !(sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL)
						&& sinfo->by == localkey
						&& mntSigOKWithRegexpOnSigner( sig, key )
						&& mntSigIsValidTime(timenow, sinfo->tstamp,
											 sinfo->sigvalidity) ) {
					/* Have an earlier signature we are looking for */
					return TRUE;
				}
			}
		}
	}
	return FALSE;
}

/*
 * See if the specified key is in the special "pairing" relationship
 * with the specified userid.  We are called with key signing userid.
 * The main point is whether key has a userid which matches.  If so,
 * we return that userid.  It is also necessary that userid be
 * self-signed, so we will check for that here.  The signing key must
 * be valid as well.
 */
static PGPKeyDBObj *
sPairedUserID (PGPKeyDBObj *userid, PGPKeyDBObj *key)
{
	PGPKeyDBObj			*keyuserid;	/* Userids on key */
	PGPKeyDBObj			*useridkey;	/* Key parent of userid */
	PGPKeyDBObj			*useridsig;	/* Sigs on userid */
	PGPByte const		*smatchuserid; /* Actual userid of *userid */
	PGPByte const		*suserid;		/* Actual userid of *keyuserid */
	PGPSize			 	lmatchuserid; /* Length of smatchuserid */
	PGPSize				luserid;		/* Length of suserid */
	PGPKeyInfo			*kinfo;
	PGPSigInfo			*uidsiginfo;
	
	pgpAssert (OBJISKEY(key));
	pgpAssert (OBJISUSERID(userid));

	kinfo = pgpKeyToKeyInfo( key );

	/* Check for key validity */
	if( kinfo->trust & (PGP_KEYTRUSTF_REVOKED | PGP_KEYTRUSTF_EXPIRED) )
		return NULL;
	
	/* Check that userid is self-signed */
	useridkey = userid->up;
	pgpAssert (OBJISKEY(useridkey));
	for (useridsig = userid->down; IsntNull(useridsig);
		 							useridsig = useridsig->next) {
		if( !pgpKeyDBObjIsReal( useridsig ) )
			continue;
		if( !OBJISSIG(useridsig) )
			continue;
		uidsiginfo = pgpSigToSigInfo( useridsig );
		/* XXX Should check expiration */
		if( (uidsiginfo->trust & PGP_SIGTRUSTF_CHECKED)
			&& !(uidsiginfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL)
			&& (uidsiginfo->type & 0xF0) == PGP_SIGTYPE_KEY_GENERIC ) {
			break;				/* Exit with useridsig nonnull on success */
		}
	}
	if( IsNull(useridsig) )			/* NULL useridsig means no self sig */
		return NULL;
		
	/* Get userid string of userid */
	smatchuserid = pgpFetchObject( userid, &lmatchuserid );

	/* See if any userids on keyuserid match */
	for (keyuserid = key->down; IsntNull(keyuserid);
		 							keyuserid = keyuserid->next) {
		if( !pgpKeyDBObjIsReal( keyuserid ) )
			continue;
		if( !OBJISUSERID(keyuserid) )
			continue;
		suserid = pgpFetchObject( keyuserid, &luserid );
		if( luserid == lmatchuserid &&
			pgpMemoryEqual( suserid, smatchuserid, luserid ) ) {
			/* Have a matching userid! */
			return keyuserid;
		}
	}
	return NULL;
}


/*
 * Find all the paths from the starting userid back to an axiomatically
 * trusted key.
 * pred holds the path which leads to the starting key from some target.
 * lastseg points at the last segment of the path, which already has
 * the key this userid is on as the dest.  We just have to fill it in
 * and recurse, or check for buckstop keys.
 *
 * We reject paths which are to keys which have
 * already been processed as signing this key.  This can happen
 * if there are two userids on a key, both signed by another key.  Otherwise
 * these would be treated as two different paths, but they
 * end up being identical.  The Maurer algorithm correctly accounts for this,
 * but it is a waste.  What we do is, for each signature, we check
 * all our earlier userid siblings and see if they have a signature from the
 * same key.  If so we skip this si
 *
 * Returns	tail pointer for updated PathList
 *
 * userid	starting userid object for search backwards through web of trust
 * ppath	tail pointer of PathList to add our new paths to
 * predh	pointer to the (head of the) Path we will add to
 * predt	tail pointer to the Path we will add to (&newseg->tail)
 * depth	length of path so far, counting newseg (e. 1 at outer level)
 * maxdepth	maximum length of paths we allow
 * timenow	current timestamp
 * db		PGPKeyDB to use for memory allocations
 * skipsigs list of signatures, chained by next, not to follow key of
 *
 */
static PathList **
sFindPathsUserID (PGPKeyDBObj *userid, PathList **ppath,
	Path **predh, Path **predt, unsigned depth,
    unsigned maxdepth, PGPUInt32 timenow, PGPKeyDB *db, PGPKeyDB *altdb,
	PGPKeyDBObj *skipsigs)
{
	PGPKeyDBObj     *sig,
	               *sigkey,
	               *localsigkey,
				   *paireduserid,
				   *ssig;
	Path		   *newseg;
	Path          **newpredt;
	double			confidence;
	PGPSigInfo	   *sinfo;
	PGPSigInfo	   *ssinfo;
	PGPKeyInfo	   *sigkeyinfo;
	int				tryalt;

	for( sig = userid->down; sig; sig = sig->next )
	{
		if( !pgpKeyDBObjIsReal( sig ) )
			continue;
		if( !OBJISSIG(sig) )
			continue;
		sinfo = pgpSigToSigInfo( sig );
		/* Skip if invalid sig */
		if( !(sinfo->trust & PGP_SIGTRUSTF_CHECKED)
			|| (sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL)
			|| !mntSigIsValidTime(timenow, sinfo->tstamp, sinfo->sigvalidity)
			|| (sinfo->type & 0xF0) != PGP_SIGTYPE_KEY_GENERIC )
			continue;
		/* Try tracing sigs back through altdb too */
		for( tryalt = 0;
			 tryalt <= (IsntNull(altdb) && (PGPPeekKeyDBObjKeyDB(sig)!=altdb));
			 ++tryalt )
		{
			if( tryalt == 0 )
			{
				sigkey = sinfo->by;
				localsigkey = sigkey;
			} else {
				PGPKeyID keyID;
				sigkey = NULL;
				if( IsntPGPError( pgpGetKeyIDOfCertifier( sig, &keyID) ) )
					pgpGetKeyByKeyID(altdb, &keyID, TRUE, FALSE, &sigkey );
				if( IsNull( sigkey ) )
					continue;
				localsigkey = sinfo->by;
			}
			/* Skip if signing key not trusted */
			if( !pgpKeyDBObjIsReal( sigkey ) )
				continue;
			if ( !sKeyHasConfidence (sigkey) )
				continue;
			if( !mntSigOKWithRegexpOnSigner( sig, sigkey ) )
				continue;

			/* Skip if there is a newer valid sig by the same key */
			if( sNewestGoodSig (userid, localsigkey) != sig )
				continue;
			/* Skip if we have already done this key on an earlier userid*/
			/* (I.e. sigkey signed more than one userid on this key) */
			if( depth>1 && sEarlierValidSig (userid, sigkey, localsigkey,
											 timenow) )
				continue;
			/*
			 * Skip if signing key is in skipsigs list.  This is for the
			 * "paired keys" feature.
			 */
			for (ssig = skipsigs; ssig; ssig = ssig->next) {
				if( !pgpKeyDBObjIsReal( ssig ) )
					continue;
				ssinfo = pgpSigToSigInfo( ssig );
				if( OBJISSIG(ssig) ) {
					if( localsigkey == ssinfo->by )
					{
						/* Exit from loop with ssig nonnull on match */
						break;
					}
				}
			}
			if( IsntNull(ssig) )
			{
				/* Non-null ssig means it was on skip list */
				continue;
			}

			/* OK, investigate this signer further */
			sigkeyinfo = pgpKeyToKeyInfo( sigkey );

			/* Don't count revoked or expired keys (new test!) */
			if( sigkeyinfo->trust & (PGP_KEYTRUSTF_REVOKED |
									 PGP_KEYTRUSTF_EXPIRED) )
				continue;

			if( (sigkeyinfo->trust & PGP_KEYTRUSTF_BUCKSTOP)
				/*&& !(sigkeyinfo->trust & PGP_KEYTRUSTF_DISABLED)*/ ) {
				/* Found a path */
				ppath = pathListAddPathClone (ppath, *predh, db);
				if( IsNull(ppath) ) {
					/* Out of memory */
					return NULL;
				}
			} else if( depth < maxdepth &&
					   !LOOKINGAT(sigkeyinfo) &&
					   (confidence = pathKeyConfidence(sigkey)) != 0 ) {
				/* Recurse */
				newseg = pathAlloc(db);
				if( IsNull(newseg) ) {
					/* Out of memory */
					return NULL;
				}
				newseg->dest = userid->up;
				newseg->src = sigkey;
				newseg->next = NULL;
				newseg->confidence = confidence;
				newpredt = pathAddSeg (predt, newseg);
				ppath = sFindPathsKey (sigkey, ppath, predh, newpredt,
						 depth+1, maxdepth, timenow, db, altdb);
				pathFree (newseg, db);
				*predt = NULL;
				if( IsNull(ppath) ) {
					/* Out of memory */
					return NULL;
				}
			} else if( depth < maxdepth &&
					   !LOOKINGAT(sigkeyinfo) &&
					   (paireduserid =
							sPairedUserID(userid, sigkey)) != NULL ) {
				/*
				 * We have a signature by a sibling key, one which matches our
				 * userid.  Recurse from that userid.  The resulting path element
				 * will have as source, signer of that paired userid, and as
				 * destination, this key.  Even though there is no "actual"
				 * signature in such a form, there is an implied signature.
				 * We won't increment depth in this case.
				 * Here we use the skipsigs feature - we don't want to follow
				 * any sigs from our pair which we are also following here.
				 * This is the only place it is used (at this writing!).
				 */
				newseg = pathAlloc(db);
				if( IsNull(newseg) ) {
					/* Out of memory */
					return NULL;
				}
				newseg->dest = userid->up;
				newseg->src = sigkey;
				newseg->next = NULL;
				newseg->confidence = 1.0;
				newpredt = pathAddSeg (predt, newseg);
				ppath = sFindPathsUserID (paireduserid, ppath, predh, newpredt,
					depth, maxdepth, timenow, db, altdb, userid->down);
				pathFree (newseg, db);
				*predt = NULL;
				if( IsNull(ppath) ) {
					/* Out of memory */
					return NULL;
				}
			}
		}
	}
	return ppath;
}


/*
 * Find all the paths from the starting key back to an axiomatically
 * trusted key.
 *
 * Returns	tail pointer for updated PathList
 *
 * start	starting key object for search backwards through web of trust
 * ppath	tail pointer of PathList to add our new paths to
 * predh	pointer to the (head of the) Path we will add to
 * predt	tail pointer to the Path we will add to
 * depth	length of path, counting segment added in sFindPathsUserID
 * maxdepth	maximum length of paths we allow
 * timenow	current timestamp
 * db		PGPKeyDB to use for memory allocations
 *
 */
static PathList **
sFindPathsKey (PGPKeyDBObj *start, PathList **ppath,
	Path **predh, Path **predt, unsigned depth,
    unsigned maxdepth, PGPUInt32 timenow, PGPKeyDB *db, PGPKeyDB *altdb)
{
	PGPKeyDBObj     *userid;
	PGPBoolean		firstuserid;
	PGPKeyInfo *	kinfo;
	PGPUserIDInfo *	uinfo;

	/* It's a key but not axiomatic */
	pgpAssert (OBJISKEY(start));
	kinfo = pgpKeyToKeyInfo( start );
	pgpAssert (!(kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP));

	SETLOOKINGAT(kinfo);
	firstuserid = TRUE;
	for (userid = start->down; IsntNull(userid); userid = userid->next) {
		if( !pgpKeyDBObjIsReal( userid ) )
			continue;
		if( !OBJISUSERID(userid) )
			continue;
		uinfo = pgpUserIDToUserIDInfo( userid );
		if( (uinfo->confidence != 0 &&
			 uinfo->confidence != PGP_NEWTRUST_UNDEFINED) ||
			(kinfo->signedTrust != 0 && firstuserid) ) {
			ppath = sFindPathsUserID (userid, ppath, predh, predt,
						depth, maxdepth, timenow, db, altdb, NULL);
			if( IsNull(ppath) )			/* out of memory */
				break;
		}
		firstuserid = FALSE;
	}
	CLEARLOOKINGAT(kinfo);
	return ppath;
}



/*
 * Find all paths backwards from the specified target userid to some key
 * which is axiomatically trusted.
 *
 * Returns	tail pointer for new PathList
 *
 * userid	starting userid object for search backwards through web of trust
 * ppath	address of PathList pointer we set to point to head of PathList
 * maxdepth	maximum length of paths we allow
 * timenow	current timestamp
 * db		PGPKeyDB to use for memory allocations
 *
 */
static PathList **
sFindPathsBack (PGPKeyDBObj *userid, PathList **ppath, unsigned maxdepth,
	PGPUInt32 const timenow, PGPKeyDB *db, PGPKeyDB *altdb)
{
    Path           *phead = NULL,
                  **ptail = &phead;

	pgpAssert (OBJISUSERID(userid));
	/* Don't SETLOOKINGAT here; must allow use of endpoint key as signer */
	ppath = sFindPathsUserID (userid, ppath, &phead, ptail, 1, maxdepth,
		      timenow, db, altdb, NULL);
	return ppath;
}

/*
 * Test whether the given key has at least one userid with defined
 * confidence, or is confident by being signed by meta introducers, or
 * is confident by being axiomatic.
 */
static PGPBoolean
sKeyHasConfidence (PGPKeyDBObj *key)
{
	PGPKeyDBObj		*uid;
	PGPKeyInfo *	kinfo;
	PGPUserIDInfo *	uinfo;

	pgpAssert (OBJISKEY(key));

	if( pgpKeyAxiomatic (key) )
		return TRUE;

	kinfo = pgpKeyToKeyInfo( key );
	if( kinfo->signedTrust != 0 )
		return TRUE;

	/* True if any userids have defined confidence */
	for (uid = key->down; IsntNull(uid); uid = uid->next) {
		if( !pgpKeyDBObjIsReal( uid ) )
			continue;
		if( !OBJISUSERID(uid) )
				continue;
		uinfo = pgpUserIDToUserIDInfo( uid );
		if( uinfo->confidence != PGP_NEWTRUST_UNDEFINED &&
				   uinfo->confidence != 0 )
			return TRUE;
	}
	return FALSE;
}

#if 0
/*
 * Test whether the given userid has defined confidence, or is confident
 * because the parent key is axiomatic or signed by meta introducers.
 */
static PGPBoolean
sUserIDHasConfidence (PGPKeyDBObj *userid)
{
	PGPKeyDBObj		*key;
	PGPKeyInfo *	kinfo;
	PGPUserIDInfo *	uinfo;

	pgpAssert (OBJISUSERID(userid));

	key = userid->up;
	pgpAssert (OBJISKEY(key));

	if( pgpKeyAxiomatic (key) )
		return TRUE;

	kinfo = pgpKeyToKeyInfo( key );
	if( kinfo->signedTrust != 0 )
		return TRUE;

	uinfo = pgpUserIDToUserIDInfo( userid );
	if( uinfo->confidence != PGP_NEWTRUST_UNDEFINED &&
					uinfo->confidence != 0 )
		return TRUE;

	return FALSE;
}
#endif


/******************** Trust Model 2 Helpers *************************/



	PGPUInt16 
pgpKeyCalcTrust(PGPKeyDBObj *key)
{
	PGPUInt16 trust;
	double confidence;

	pgpAssert(OBJISKEY(key));

	confidence = pathKeyConfidence (key);
	if( confidence >= 1. )
		trust = PGP_TRUST_INFINITE;
	else
		trust = pgpDoubleToTrust (1. / (1.-confidence));
	return trust;
}
	       



/* Meta-Introducer and trust signature support */



/*
 * Given a signature of level >= 1, a current trust level, and an allowed
 * further recursion depth, calculate the signedTrust levels for the key,
 * and propagate further if level > 1.  depthLimit tells how many levels
 * max we can propagate, counting this as one.  That is used to enforce
 * both a maximum limit on how deep this process goes, and for each
 * MI/MMI etc. to limit how far forward trust goes depending on the
 * amount of "meta", that is, the level.
 *
 * A level of 0 means an ordinary signature; those aren't considered here.
 * A level of 1 means a trust signature, typically issued by a meta-
 * introducer, but also possibly by an axiomatic key.  Those are applied
 * here to form the signedTrust level on the key.  A simple maximum is
 * used to deal with multiple signed trusts on the key (e. if both a
 * MI and the user specified trust on the key).
 * A level of 2 means a meta introducer.  Any trust signatures by him
 * get propagated via one level of recursion.
 * Higher levels allow more levels of recursion.
 *
 * We apply several simplifications which future versions may enhance.
 * We only allow 2nd-level (meta and above) propagation if fully
 * trusted.  That is, meta introducers, and the user himself, can specify
 * partial trust levels.  But the MI, or an MMI, etc., must be given
 * full trust.  In this trust model we can't easily handle gracefully
 * degrading trust.
 */
static void
mntApplyTrustSig (PGPKeyDBObj const *sig, PGPByte trustValue,
				  PGPUInt32 depthLimit, int thresh, PGPUInt32 timenow)
{
	PGPKeyDBObj		*key;
	PGPKeyInfo *	kinfo;
	PGPSigInfo *	sinfo;
	PGPByte			trustVal;

	pgpAssert( OBJISSIG( sig ) );
	sinfo = pgpSigToSigInfo( sig );

	/* First calculate signed trust on this key. */

	/* Don't count nonexportable sigs by others */
	if( mntForeignSignature(sig) )
		return;

	/* For now support only full trust on meta introducer signatures */
	if( trustValue < thresh && sinfo->trustLevel >= 2 )
		return;

	/*
	 * See if trust signature is ruled out by regexp.
	 * (Note that this looks not at the sig's regexp, but at any on the
	 * signing key.  E. a meta introducer may include a regexp on a sig
	 * it issues.)
	 */
	if( !mntSigOKWithRegexp( sig )  )
		return;

	/* Find signed key */
	key = sig->up;
	while (!OBJISTOPKEY(key))
		key = key->up;
	
	kinfo = pgpKeyToKeyInfo( key );

	/*
	 * For now only use the maximum value of signed trusts, don't try
	 * to accumulate them.  Thorny problems!
	 */
	trustVal = sinfo->trustValue;

	/*
	 * Mark trust value.  If ours is used, copy any regexp associated with
	 * the signature.
	 * Have some heuristics to compare regexp and non-regexp sigs.
	 */
	if( SIGUSESREGEXP( sinfo ) ) {
		if( kinfo->signedTrust > 0 && IsNull(kinfo->regexp) ) {
			/* Key already has some non-regexp trust, so leave it */
		} else if( trustVal > kinfo->signedTrust ) {
			/* Key had no trust or regexp trust, and we are stronger */
			void *rexp = pgpSigRegexp( sig );
			/* Abort this chain if some problem parsing regexp */
			if( IsNull(rexp) )
				return;
			kinfo->signedTrust = trustVal;
			kinfo->regexp = rexp;
		}
	} else {
		if( IsntNull(kinfo->regexp) || trustVal > kinfo->signedTrust ) {
			/*
			 * Key had regexp trust, but we don't; or key had no regexp trust,
			 * but we are stronger.
			 */
			kinfo->regexp = NULL;
			kinfo->signedTrust = trustVal;
		}
	}
	
	/*
	 * Then, if level > 1 and we have not reached depthLimit,
	 * propagate forwards
	 */

	if( sinfo->trustLevel > 1 && depthLimit > 1 ) {
		PGPKeyDBObj const *keysig = kinfo->sigsby;
		while( IsntNull(keysig) ) {
			PGPKeyDBObj *newestsig;
			PGPSigInfo *newestsinfo;

			pgpAssert( OBJISSIG( keysig ) );
			newestsig = mntNewestSiblingSig(&keysig);
			if( !mntSigIsValid( newestsig, timenow ) )
				continue;
			newestsinfo = pgpSigToSigInfo( newestsig );

			/* Skip if not a trust sig */
			if( newestsinfo->trustLevel < 1 )
				continue;
			/* Skip if not a valid key sig */
			if( !mntSigIsValidTimeKeySig (newestsig, timenow) )
				continue;
			/*
			 * Here we have a good trust or metaintroducer si
			 * Propagate it forward.
			 */
			mntApplyTrustSig (newestsig, sinfo->trustValue,
					 pgpMin(sinfo->trustLevel-1, (PGPByte)depthLimit-1),
					 thresh, timenow);
		}
	}
}


/******************** Trust Propagation Main *************************/



/*
 * Run the trust propagation pass on the keyrin
 * If changedSet is not null, add keys to it that we change
 */
	PGPError
pgpPropagateTrustInternal(PGPKeySet *set, PGPKeyDB *altdb,
	PGPUInt32 const timenow, PGPKeySet *changedSet)
{
	PGPKeyDB *db;
	PGPContextRef context;
	PGPEnv *pgpEnv;
	PGPUInt32 certdepth;
	PGPKeyDBObj const *list;
	PGPKeyDBObj *key, *child;
	PGPKeyDBObj *sig;
	PGPKeyInfo *kinfo;
	PGPKeyInfo *axkeyinfo;
	PGPKeyInfo *kinfo2;
	PGPUserIDInfo *uinfo;
	PGPSigInfo *sinfo;
	PGPSigInfo *sinfo2 = NULL;
	PGPSigInfo *newestsiginfo;
	unsigned tmptrust;
	unsigned t;
	unsigned long count;	/* Count of keys needing trust assigned */
	PGPUInt16 validity;
	PGPKeyDBObj const *axkey;
	PathList       *pathlist = NULL;
	PGPKeyDBObj     *userid;
	double          confidence;
	PGPBoolean		bChanged = FALSE;

	/* Get depth we will follow */
	db = PGPPeekKeySetKeyDB( set );
	context = PGPPeekKeyDBContext( db );
	pgpEnv = pgpContextGetEnvironment( context );
	certdepth = pgpenvGetInt(pgpEnv, PGPENV_CERTDEPTH, NULL, NULL);

	/*
	 * Preprocessing: reset all userid trusts to 0, and initialize list
	 * of trusted keys to the axiomatic ones.
	 */
	list = NULL;
	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, set ) )
			continue;
		pgpAssert(OBJISKEY(key));
		kinfo = pgpKeyToKeyInfo( key );
		kinfo->signedTrust = 0;
		CLEARLOOKINGAT(kinfo);
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !pgpKeySetIsMember( child, set ) )
				continue;
			if( OBJISUSERID(child) ) {
				uinfo = pgpUserIDToUserIDInfo( child );
				uinfo->valid = 0;
			}
		}
		/*
		 * Find each key's revoked status - find
		 * the most important self-signature.
		 */
		sig = 0;
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !pgpKeySetIsMember( child, set ) )
				continue;
			/*
			 * Ignore sigs by others, unchecked sigs,
			 * postdated sigs, and expired sigs.
			 * @@@ TODO: change the CHECKED and TRIED bits
			 * For untried revocation sigs, accept them if key's
			 * trust bits already show it as revoked.  PGP 2 does
			 * not store trust packets on revocation signatures, and so
			 * neither does this library, to avoid crashing older PGP's
			 * when they are run on our keyrings.
			 */
			if( OBJISSIG(child) ) {
				sinfo = pgpSigToSigInfo( child );
				if( mntSigIsValidTime(timenow, sinfo->tstamp,
									  sinfo->sigvalidity)) {
					/* Any revocation certificate wins, period. */
					if( sigRevokesKey(child) ) {
						sig = child;
						sinfo2 = sinfo;
						break;
					}
					/* Among other checked selfsigs, choose the most recent */
					if( sinfo->by == key
						&& (sinfo->trust & PGP_SIGTRUSTF_CHECKED) ) {
						if( IsNull(sig) || (sinfo2->tstamp < sinfo->tstamp) ) {
							sig = child;
							sinfo2 = sinfo;
						}
					}
				}
			}
		}
		sinfo = sinfo2;

		/* At this point sig holds most recent sig, or first revocation
		 * sig found, which is directly on the key.
		 */

		tmptrust = kinfo->trust;
		/* Revocation is a function of revocation certificates.
		   Make sure axiomatic (buckstop) bit is not set. */
		if( IsntNull(sig) && sinfo->type == PGP_SIGTYPE_KEY_REVOKE ) {
			tmptrust &= ~(PGP_KEYTRUSTF_BUCKSTOP | kPGPKeyTrust_Mask);
			tmptrust |= kPGPKeyTrust_Never | PGP_KEYTRUSTF_REVOKED;
		}
		else
			tmptrust &= ~PGP_KEYTRUSTF_REVOKED;

		/* Check if key has expired */
		if( kinfo->creationtime < timenow && kinfo->validityperiod > 0 &&
			 (unsigned)((timenow - kinfo->creationtime)/86400) >=
										kinfo->validityperiod ) {
			tmptrust &= ~(PGP_KEYTRUSTF_BUCKSTOP | kPGPKeyTrust_Mask);
			tmptrust |= kPGPKeyTrust_Never | PGP_KEYTRUSTF_EXPIRED;
		}
		else
			tmptrust &= ~PGP_KEYTRUSTF_EXPIRED;

		if( kinfo->trust != tmptrust ) {
			kinfo->trust = (PGPByte)tmptrust;
			bChanged = TRUE;
			if( IsntNull( changedSet ) )
				PGPAddKey( key, changedSet );
		}

		/*  Check any subkeys for revocation or expiration.  */
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !pgpKeySetIsMember( child, set ) )
				continue;
		    if( OBJISSUBKEY(child) ) {
				kinfo2 = pgpKeyToKeyInfo( child );
			    tmptrust = kinfo2->trust;
				tmptrust &= ~(PGP_KEYTRUSTF_REVOKED | PGP_KEYTRUSTF_EXPIRED);
				for (sig = child->down; IsntNull(sig); sig = sig->next) {
					if( !pgpKeyDBObjIsReal( sig ) )
						continue;
					if( !pgpKeySetIsMember( sig, set ) )
						continue;
			        if( OBJISSIG(sig) ) {
						sinfo = pgpSigToSigInfo( sig );
						if( sigRevokesKey(sig) &&
							mntSigIsValidTime (timenow, sinfo->tstamp,
											   sinfo->sigvalidity) ) {
							tmptrust |= PGP_KEYTRUSTF_REVOKED;
							break;
						}
					}
				}
				if( kinfo2->creationtime < timenow &&
						kinfo2->validityperiod > 0 &&
						(unsigned)((timenow - kinfo2->creationtime)/86400) >= 
						kinfo2->validityperiod )
			        tmptrust |= PGP_KEYTRUSTF_EXPIRED;
				
				if( kinfo2->trust != tmptrust ) {
				    kinfo2->trust = (PGPByte) tmptrust;
					if( IsntNull( changedSet ) )
						PGPAddKey( key, changedSet );
					bChanged = TRUE;
				}
			}
		}

		/* Search for axiomatic keys at root of graph.  Buckstop bit must be 
		   set, and key must not be revoked or expired. */
		if( kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP
		    && !(kinfo->trust & (PGP_KEYTRUSTF_REVOKED|PGP_KEYTRUSTF_EXPIRED))
/*		    && !(kinfo->trust & PGP_KEYTRUSTF_DISABLED) */	 )
		{
		    /* Make sure key marked as axiomatic has a secret component.
			   (This may not be true if a public keyring is opened, but
			   *not* its corresponding secret keyring). If not, reset
			   the buckstop bit and set trust to 'never'. */
		    tmptrust = kinfo->trust & ~PGP_KEYTRUSTF_BUCKSTOP;
			if( pgpKeyIsSec( key ) )
				tmptrust |= PGP_KEYTRUSTF_BUCKSTOP;
			if( !(tmptrust & PGP_KEYTRUSTF_BUCKSTOP) ) {
			    /* Don't have private key */  
			    tmptrust &= ~kPGPKeyTrust_Mask;
				tmptrust |= kPGPKeyTrust_Never;
			    kinfo->trust = tmptrust;
				if( IsntNull( changedSet ) )
					PGPAddKey( key, changedSet );
				bChanged = TRUE;
			} else {
			    /* Yes, we do have private key */ 
				/* Link into list via util field */
				kinfo->util = (PGPKeyDBObj *)list;
				list = key;
				/* Make sure all userids on axiomatic key are fully valid */
				/* XXX (Really should only do this on self-signed userids! */
				for (child = key->down; IsntNull(child);
					 							child = child->next) {
					if( !pgpKeyDBObjIsReal( child ) )
						continue;
					if( !pgpKeySetIsMember( child, set ) )
						continue;
				    if( OBJISUSERID(child) ) {
						uinfo = pgpUserIDToUserIDInfo( child );
					    uinfo->valid = (PGPUInt16) PGP_TRUST_INFINITE;
					}
				}
			}
		}
	}

	if( IsntNull( altdb ) )
	{
		/* Also add axiomatic keys from backup DB to the list for meta
		 * introducer trust purposes
		 */
		for (key = altdb->firstKeyInDB; IsntNull(key); key = key->next) {
			if( !pgpKeyDBObjIsReal( key ) )
				continue;
			kinfo = pgpKeyToKeyInfo( key );
			if( kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP
				&& !(kinfo->trust & (PGP_KEYTRUSTF_REVOKED
									 |PGP_KEYTRUSTF_EXPIRED))
	/*		    && !(kinfo->trust & PGP_KEYTRUSTF_DISABLED) */	 )
			{
				if( pgpKeyIsSec( key ) )
				{
					kinfo->util = (PGPKeyDBObj *)list;
					list = key;
				}
			}
		}
	}

	/* The actual trust computation */

	pgpKeyDBListSigsBy(db);			/* Canonicalize sigsby pointers */

	/* Propagate meta introducer trust forwards first */
	/* At this point, "list" contains axiomatic keys */
	
	for (axkey = list; IsntNull(axkey)
			 		   && IsntNull(axkeyinfo = pgpKeyToKeyInfo(axkey));
												axkey = axkeyinfo->util) {
		PGPKeyDBObj const *axsig = axkeyinfo->sigsby;
		while (IsntNull(axsig)) {
			PGPKeyDBObj *newestsig;
			pgpAssert( OBJISSIG( axsig ) );
			newestsig = mntNewestSiblingSig(&axsig);
			pgpAssert( OBJISSIG( newestsig ) );
			if( !mntSigIsValid( newestsig, timenow ) )
				continue;
			newestsiginfo = pgpSigToSigInfo( newestsig );
			/* Skip if not a trust sig */
			if( newestsiginfo->trustLevel < 1 )
				continue;
			/* Skip if not a valid key sig */
			if( !mntSigIsValidTimeKeySig (newestsig, timenow) )
				continue;
			/*
			 * Here we have a good trust or metaintroducer signature by us.
			 * Propagate it forward.
			 */
			mntApplyTrustSig (newestsig, PGP_NEWTRUST_INFINITE,
					certdepth, pgpenvCompleteTrust(pgpEnv), timenow);
		}
	}


	/* Find a path back from each userid which sets its validity */
	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, set ) )
			continue;
		pgpAssert(OBJISKEY(key));
		kinfo = pgpKeyToKeyInfo( key );

		for (userid = key->down; IsntNull(userid); userid = userid->next) {
			if( !pgpKeyDBObjIsReal( userid ) )
				continue;
			if( !OBJISUSERID(userid) )
				continue;
			if( !pgpKeySetIsMember( userid, set ) )
				continue;
			uinfo = pgpUserIDToUserIDInfo( userid );

			/*  Calculate "real" validity for userids.  If the key
				is axiomatic, this will be overidden when the 
				validity is returned by	pgpUserIDValidity. */
/* Don't automatically mark axiomatic keys' userids as trusted */
#if 0
			if( kinfo->trust & PGP_KEYTRUSTF_BUCKSTOP
				&& !(kinfo->trust &
					 (PGP_KEYTRUSTF_REVOKED|PGP_KEYTRUSTF_EXPIRED)
				/* && !(kinfo->trust & PGP_KEYTRUSTF_DISABLED) */) ) {
				uinfo->valid = PGP_TRUST_INFINITE;
				continue;
			}
#endif
			/*
			 * Find all paths back from this userid, among only trusted
			 * introducers, to an axiomatic key.
			 */
			sFindPathsBack (userid, &pathlist, certdepth,
				timenow, db, altdb);
			if( IsntNull(pathlist) ) {
				pathlist = pathListPrune (pathlist, TOTALPATHMAX, db);
#ifdef DEBUGPATH
				fprintf (stdout, "Found path for ");
				pgpDebugPrintKey (stdout, key);
				fprintf (stdout, "\n");
#endif
				confidence = pathListConfidence (pathlist, db);
				if( confidence >= 1. )
					uinfo->valid = PGP_TRUST_INFINITE;
				else
					uinfo->valid = pgpDoubleToTrust (1. / (1.-confidence));
				pathListFreeAll (pathlist, db);
				pathlist = NULL;
			}
		}
	}
	memPoolEmpty(&db->pathpool);
	db->paths = NULL;
	db->pathlists = NULL;


	/* Postprocessing: set the various trust bytes, */
	count = 0;


	for (key = db->firstKeyInDB; IsntNull(key); key = key->next) {
		PGPKeyDBObjRef altkey;
		PGPKeyInfo *altkinfo;

		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, set ) )
			continue;

		pgpAssert( OBJISKEY( key ) );
		kinfo = pgpKeyToKeyInfo( key );

		/* Find corresponding key in other db if it exists */
		altkey = NULL;
		altkinfo = NULL;
		if( IsntNull( altdb ) )
		{
			PGPKeyID keyid;
			pgpKeyID8( key, NULL, &keyid );
			pgpGetKeyByKeyID(altdb, &keyid, FALSE, FALSE, &altkey );
			if( IsntNull( altkey ) )
				altkinfo = pgpKeyToKeyInfo( altkey );
		}
		
		tmptrust = kinfo->trust & kPGPKeyTrust_Mask;

		/* Set the signature trusts to match the key */
		for (sig = kinfo->sigsby; IsntNull(sig); sig = sinfo->nextby) {
			pgpAssert( OBJISSIG( sig ) );
			pgpAssert( pgpKeyDBObjIsReal( sig ) );
			sinfo = pgpSigToSigInfo( sig );

			/* Trust is either key's, or NEVER if it's a bad sig */
			t = sinfo->trust & ~kPGPKeyTrust_Mask;
			if( sinfo->trust & PGP_SIGTRUSTF_CHECKED )
				t |= tmptrust;
			else
				t |= kPGPKeyTrust_Never;
			if( sinfo->trust != t )
			{
				sinfo->trust = t;
				if( IsntNull( changedSet ) )
					PGPAddKey( PGPPeekKeyDBObjKey(sig), changedSet );
				bChanged = TRUE;
			}
		}

		/* Update each userid's validity byte */
		for (child = key->down; IsntNull(child); child = child->next) {
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !OBJISUSERID(child) )
				continue;
			if( !pgpKeySetIsMember( child, set ) )
				continue;
			uinfo = pgpUserIDToUserIDInfo( child );
			if( kinfo->trust & PGP_KEYTRUSTF_REVOKED )
				validity = 0;
			else if( uinfo->valid == PGP_TRUST_INFINITE )
				validity = PGP_NEWTRUST_INFINITE;
			else {
				validity = uinfo->valid >> TRUST_CERTSHIFT;
				if( validity > PGP_NEWTRUST_MAX )
					  validity = PGP_NEWTRUST_MAX;
			}

			/* Take max of validity from this and from other keydb */
			if( IsntNull( altkey ) ) {
				PGPByte const *uidbuf;
				PGPSize uidbuflen;
				PGPKeyDBObjRef altchild;

				uidbuf = pgpFetchObject( child, &uidbuflen );
				altchild = pgpFindMatchingChild( altkey, uidbuf, uidbuflen );
				if( IsntNull( altchild ) ) {
					PGPUserIDInfo *altchildinfo;
					altchildinfo = pgpUserIDToUserIDInfo( altchild );
					validity = pgpMax( validity, altchildinfo->validity );
				}
			}

			if( validity != uinfo->validity ) {
				uinfo->validity = (PGPByte) validity;
				if( IsntNull( changedSet ) )
					PGPAddKey( key, changedSet );
				bChanged = TRUE;
			}
			tmptrust = pgpTrustExternToOld( pgpEnv, (PGPUInt8) validity );
			tmptrust = pgpTrustOldKeyToUserID( tmptrust );
			t = tmptrust | (uinfo->oldvalidity &
						(PGP_USERIDTRUSTF_WARNONLY|PGP_USERIDTRUSTF_PRIMARY));
			if( t != uinfo->oldvalidity ) {
				uinfo->oldvalidity = t;
				if( IsntNull( changedSet ) )
					PGPAddKey( key, changedSet );
				bChanged = TRUE;
			}

			/* Reset valid with adjusted value. */
			if( uinfo->validity == PGP_NEWTRUST_INFINITE )
				uinfo->valid = PGP_TRUST_INFINITE;
			else
				uinfo->valid = 
				    uinfo->validity << TRUST_CERTSHIFT;
		}
	}

	/* Clean up after regexp processing */
	mntClearKeyRegexps( db );

	if( bChanged )
		pgpKeyDBChanged( db, TRUE );

	/* Et voila, we're done! */
	return kPGPError_NoErr;
}


	
	PGPError
PGPCalculateTrust( PGPKeySetRef set, PGPKeyDBRef altdb )
{
	PGPKeyDBRef setdb;
	PGPUInt32 *keylist;
	PGPSize keylistsize;
	PGPError err;

	PGPValidateKeySet( set );
	if( IsntNull( altdb ) )
		PGPValidateKeyDB( altdb );

	pgpEnterPGPErrorFunction();

	setdb = PGPPeekKeySetKeyDB( set );
	if( setdb == altdb )
		altdb = NULL;
	if( pgpFrontEndKeyDB( setdb ) )
	{
		PGPContextRef context = PGPPeekKeyDBContext( setdb );
		PGPUInt32 *changedkeylist;
		PGPSize changedkeylistsize;

		err = pgpKeySetFlatten( set, &keylist, &keylistsize );
		if( IsPGPError( err ) )
			return err;
		err = pgpPropagateTrust_back( context, setdb->id, keylist, keylistsize,
									  (IsNull(altdb)?0:altdb->id),
									  PGPGetTime(), &changedkeylist,
									  &changedkeylistsize);
		if( IsPGPError( err ) )
			return err;
		if( changedkeylistsize != 0 )
			pgpKeyDBChanged (setdb, TRUE);
		return pgpKeyRefreshFromList( setdb, changedkeylist,
									  changedkeylistsize );
	}

	return pgpPropagateTrustInternal( set, altdb, PGPGetTime(), NULL );
}



/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
